﻿using Inet.Viewer.Data;
using Inet.Viewer.Resources;
/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace Inet.Viewer.WinForms.Export
{
    /// <summary>
    /// Settings panel for exporting a report as image(s). The user can choose the
    /// image format various scaling and coloring options.
    /// </summary>
    public partial class ImageSettingsControl : FormatSettingsControl
    {
        private const string PropImageSize = "imagesize";
        private const string PropBackground = "background";
        private const string PropZoom = "zoom";
        private const string PropWidth = "width";
        private const string PropHeight = "height";

        private readonly string[] Formats = { "JPG", "PNG", "BMP", "GIF" };

        private readonly Regex percentRegex = new Regex(@"^\d+%$");
        private ColorDialog colorDialog = new ColorDialog();
        private bool multiPageReport = false;

        /// <summary>
        /// Creates the control instance.
        /// </summary>
        public ImageSettingsControl()
        {
            InitializeComponent();
            cbScaleFactor.SelectedIndex = 2;
        }

        /// <inheritdoc />
        public override Bitmap Icon { get { return FormatResource.img_48; } }

        /// <inheritdoc />
        public override string Label { get { return strings.ExportImage; } }

        /// <inheritdoc />
        public override string FileSuffix
        {
            get
            {
                return multiPageReport ? GetSelectedFormat() + ".zip" : GetSelectedFormat();
            }
        }

        /// <inheritdoc />
        public override void CollectExportParameters(Dictionary<string, string> exportParams)
        {
            exportParams[URLRenderData.ParameterExportFmt] = GetSelectedFormat();

            if (tbColor.Text.Length != 0)
            {
                exportParams[PropBackground] = tbColor.Text;
            }

            if (rbScaleFactor.Checked)
            {
                exportParams[PropZoom] = cbScaleFactor.Text;
            }
            else if (rbWidth.Checked)
            {
                exportParams[PropWidth] = numWidth.Value.ToString();
            }
            else
            {
                exportParams[PropHeight] = numWidth.Value.ToString();
            }
        }

        /// <inheritdoc />
        public override bool MultiPageReport
        {
            set
            {
                multiPageReport = value;
            }
        }

        /// <summary>
        /// Returns the currently selected image format as 3-character string.
        /// </summary>
        /// <returns>the currently selected image format as 3-character string</returns>
        private string GetSelectedFormat()
        {
            if (cbFormat.SelectedItem == null)
            {
                return "";
            }
            string selItem = cbFormat.SelectedItem.ToString().ToLower();
            int i = selItem.IndexOf(' ');
            return i == -1 ? selItem : selItem.Substring(0, i);
        }

        /// <summary>
        /// Called when the user clicks on the color chooser button. Opens a
        /// color dialog.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnColorPicker_Click(object sender, EventArgs e)
        {
            try
            {
                int color;
                if (tbColor.Text.StartsWith("0x"))
                {
                    color = Convert.ToInt32(tbColor.Text.Substring(2), 16);
                }
                else
                {
                    color = Convert.ToInt32(tbColor.Text);
                }
                colorDialog.Color = Color.FromArgb((color & 0xFF) << 16 | color & 0xFF00 | (color >> 16) & 0xFF);
            }
            catch (Exception)
            {
                // something went wrong with string-to-int conversion
                // just ignore the original value
            }
            if (colorDialog.ShowDialog() == DialogResult.OK)
            {
                int rgb = colorDialog.Color.ToArgb();
                tbColor.Text = "0x" + ((rgb & 0xFF) << 16 | rgb & 0xFF00 | (rgb >> 16) & 0xFF).ToString("X6");
            }
        }

        /// <summary>
        /// Called when the state of a radio button changed. Enables/disables 
        /// the corresponding fields.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void rb_CheckedChanged(object sender, EventArgs e)
        {
            cbScaleFactor.Enabled = rbScaleFactor.Checked;
            numWidth.Enabled = rbWidth.Checked;
            numHeight.Enabled = rbHeight.Checked;
            labelWidth.Enabled = rbWidth.Checked;
            labelHeight.Enabled = rbHeight.Checked;
        }

        /// <summary>
        /// Called when the user changes the image format. Fires a format
        /// changed event since the file name needs to be updated in the main
        /// dialog panel.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event's arguments</param>
        private void cbFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            OnFormatChanged();
        }

        /// <summary>
        /// Validates the user input from the field "Scale Factor".
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event's arguments</param>
        private void cbScaleFactor_Validating(object sender, CancelEventArgs e)
        {
            if (!percentRegex.IsMatch(cbScaleFactor.Text))
            {
                e.Cancel = true;
                cbScaleFactor.SelectAll();
                errorProvider1.SetError(cbScaleFactor, "Percentage value required");
            }
        }

        /// <summary>
        /// <inherticdoc/>
        /// </summary>
        override public HashSet<string> AllowedFormats
        {
            set
            {
                bool anyAllowed = false;

                cbFormat.Items.Clear();
                foreach (string format in Formats)
                {
                    if (value.Count == 0 || value.Contains(format.ToLower()))
                    {
                        cbFormat.Items.Add(format);
                        anyAllowed = true;
                    }
                }
                if (anyAllowed)
                {
                    cbFormat.SelectedIndex = 0;
                }
                Allowed = value.Count == 0 || anyAllowed;
            }
        }
    }
}
